package de.davidartmann.rosa.fragment;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.Snackbar;
import android.support.v4.app.ActivityCompat;
import android.support.v4.app.Fragment;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AlertDialog;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.EditText;
import android.widget.ImageView;

import com.squareup.picasso.Picasso;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.activity.MainListActivity;
import de.davidartmann.rosa.db.async.CreateOrUpdatePerson;
import de.davidartmann.rosa.db.async.UpdatePersonInDb;
import de.davidartmann.rosa.model.Person;
import de.davidartmann.rosa.util.RoundedTransformation;

/**
 * Fragment for the edit-view of a detailed persons view.
 * Created by david on 03.03.16.
 */
public class DetailEditFragment extends Fragment implements View.OnClickListener,
        UpdatePersonInDb.IUpdatePersonInDb {

    private static final String TAG = DetailEditFragment.class.getSimpleName();
    private static final int REQUEST_IMAGE_CAPTURE = 1;
    private static final int REQUEST_IMAGE_PICK = 2;
    private static final int REQUEST_PERMISSION_READ_EXTERNAL_STORAGE = 3;
    private static final String NEW_PERSON = "NEW_PERSON";
    private static final String NUMBER_ACTIVE_PERSONS = "NUMBER_ACTIVE_PERSONS";

    private EditText mEditTextName;
    private EditText mEditTextPhone;
    private EditText mEditTextEmail;
    private EditText mEditTextAddress;
    private EditText mEditTextPrice;
    private EditText mEditTextMisc;
    private ImageView mImageViewPicture;
    private Person mPerson;
    private boolean mIsUpdate;
    private int mNumberOfActivePersons;
    private String mCurrentPicturePath;

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_detail_edit, container, false);
        setHasOptionsMenu(true);
        mIsUpdate = false;
        mEditTextName = (EditText) view.findViewById(R.id.fragment_detail_edit_name);
        mEditTextPhone = (EditText) view.findViewById(R.id.fragment_detail_edit_phone);
        mEditTextEmail = (EditText) view.findViewById(R.id.fragment_detail_edit_email);
        mEditTextAddress = (EditText) view.findViewById(R.id.fragment_detail_edit_address);
        mEditTextPrice = (EditText) view.findViewById(R.id.fragment_detail_edit_price);
        mEditTextMisc = (EditText) view.findViewById(R.id.fragment_detail_edit_misc);
        mImageViewPicture = (ImageView) view.findViewById(R.id.fragment_detail_edit_picture);
        Bundle bundle = getArguments();
        if (bundle != null) {
            mPerson = (Person) bundle.getSerializable(DetailFragment.PERSON);
            if (mPerson != null) {
                assignData(mPerson);
                mIsUpdate = true;
            }
            boolean isNewPerson = bundle.getBoolean(NEW_PERSON);
            if (isNewPerson) {
                mIsUpdate = false;
            }
            //mNumberOfActivePersons = bundle.getInt(DetailFragment.POSITION);
            mNumberOfActivePersons = bundle.getInt(NUMBER_ACTIVE_PERSONS);
        }
        resetActionBar(true);
        mImageViewPicture.setOnClickListener(this);
        if (!mIsUpdate || (mIsUpdate && mPerson.getPictureUrl() == null)) {
            mImageViewPicture.setImageDrawable(getContext().getDrawable(R.drawable.user_add));
        }
        return view;
    }

    private void assignData(Person person) {
        Context context = getContext();
        mEditTextName.setText(person.getName());
        mEditTextPhone.setText(person.getPhone());
        mEditTextEmail.setText(person.getEmail());
        mEditTextAddress.setText(person.getAddress());
        mEditTextPrice.setText(person.getPrice());
        mEditTextMisc.setText(person.getMisc());
        int dimen;
        if (context.getResources().getConfiguration().orientation
                == Configuration.ORIENTATION_PORTRAIT) {
            dimen = R.dimen.imageview_detail_dimen_portrait;
        } else {
            dimen = R.dimen.imageview_detail_dimen_landscape;
        }
        loadRoundedPictureIntoImageView(context, person.getPictureUrl(), dimen);
        /*
        Picasso.with(context)
                .load(person.getPictureUrl())
                .placeholder(R.drawable.user_add)
                .error(R.drawable.user_add)
                .transform(new RoundedTransformation(0, dimen))
                .resizeDimen(dimen, dimen)
                .centerCrop()
                .into(mImageViewPicture);
        */
    }

    public static DetailEditFragment newInstanceForCreateOrUpdate(Person person,
                                                                  int numOfActivePersons) {
        DetailEditFragment detailEditFragment = new DetailEditFragment();
        Bundle bundle = new Bundle();
        // update Person, if null we create new person
        if (person != null) {
            bundle.putSerializable(DetailFragment.PERSON, person);
        }
        bundle.putInt(NUMBER_ACTIVE_PERSONS, numOfActivePersons);
        detailEditFragment.setArguments(bundle);
        return detailEditFragment;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        menu.clear();
        inflater.inflate(R.menu.menu_detail_edit, menu);
        MenuItem menuItem = menu.findItem(R.id.menu_detail_edit_save);
        menuItem.setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);
        menuItem.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem item) {
                showCategoryPicker();
                return true;
            }
        });
    }

    private Person getPersonFromFieldValues(int which) {
        if (mPerson == null) {
            mPerson = new Person();
        }
        // if update, we do not need to do anything, because the person simply keeps its position
        if (!mIsUpdate) {
            mPerson.setPosition(mNumberOfActivePersons+1);
        }
        mPerson.setPrice(mEditTextPrice.getText().toString().trim());
        mPerson.setActive(true);
        mPerson.setMisc(mEditTextMisc.getText().toString().trim());
        mPerson.setAddress(mEditTextAddress.getText().toString().trim());
        mPerson.setCategory(which);
        mPerson.setEmail(mEditTextEmail.getText().toString().trim());
        mPerson.setName(mEditTextName.getText().toString().trim());
        if (mCurrentPicturePath != null) {
            mPerson.setPictureUrl(mCurrentPicturePath);
        }
        mPerson.setPhone(mEditTextPhone.getText().toString().trim());
        return mPerson;
    }

    private void showCategoryPicker() {
        Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle(getString(R.string.Wähle_Kategorie));
        final ArrayAdapter<String> arrayAdapter = new ArrayAdapter<>(context,
                android.R.layout.simple_list_item_single_choice);
        arrayAdapter.addAll(getResources().getStringArray(R.array.categories));
        int category = mIsUpdate ? mPerson.getCategory() : 0;
        builder.setSingleChoiceItems(arrayAdapter, category, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new CreateOrUpdatePerson(getContext(), mIsUpdate)
                        .execute(getPersonFromFieldValues(which));
                dialog.dismiss();
                getActivity().getSupportFragmentManager().beginTransaction()
                        .replace(R.id.activity_main_framelayout, new MainListFragment())
                        .commit();
                resetActionBar(false);
            }
        });
        builder.setNegativeButton("Abbrechen", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    private void resetActionBar(boolean b) {
        ActionBar actionBar = ((MainListActivity)getActivity()).getSupportActionBar();
        if (actionBar != null) {
            actionBar.setDisplayHomeAsUpEnabled(b);
            actionBar.setHomeButtonEnabled(b);
            if (mIsUpdate) {
                actionBar.setTitle(R.string.Person_bearbeiten);
            } else {
                actionBar.setTitle(R.string.Neue_Person);
            }
        }
    }

    private Intent getPickImageIntent() {
        Intent intent = new Intent();
        intent.setType("image/*");
        intent.setAction(Intent.ACTION_GET_CONTENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        return Intent.createChooser(intent, "Select picture");
    }

    @Override
    public void onClick(View v) {
        if (deviceHasCamera()) {
            showPictureDialog();
        } else {
            showSnackBar("Leider verfügt ihr Gerät über keine Kamera", "device has no camera");
        }
    }

    private void showPictureDialog() {
        final Context context = getContext();
        String[] text = new String[] {"Mit Kamera aufnehmen", "Aus Galerie auswählen",
                "Bild löschen"};
        CharSequence[] items;
        if (mPerson == null) {
            items = new CharSequence[2];
        } else {
            items = new CharSequence[3];
        }
        System.arraycopy(text, 0, items, 0, items.length);
        AlertDialog.Builder builder	= new AlertDialog.Builder(context);
        builder.setTitle("Was möchten Sie tun?");
        ArrayAdapter<String> arrayAdapter = new ArrayAdapter<>(context,
                android.R.layout.select_dialog_item, text);
        builder.setAdapter(arrayAdapter, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                if (which == 0) {
                    Intent takePictureIntent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
                    if (takePictureIntent
                            .resolveActivity(getActivity().getPackageManager()) != null) {
                        File photoFile = createImageFile();
                        if (photoFile != null) {
                            takePictureIntent.putExtra(MediaStore.EXTRA_OUTPUT,
                                    Uri.fromFile(photoFile));
                            startActivityForResult(takePictureIntent, REQUEST_IMAGE_CAPTURE);
                            dialog.dismiss();
                        }
                    }
                } else if (which == 1) {
                    requestReadExternalStorage();
                    dialog.dismiss();
                } else {
                    deleteOldImgFile();
                    mPerson.setPictureUrl(null);
                    Picasso.with(context).load(R.drawable.ic_face_24dp).into(mImageViewPicture);
                    new UpdatePersonInDb(context, DetailEditFragment.this).execute(mPerson);
                    dialog.dismiss();
                }
            }
        });
        builder.create().show();
    }

    private void requestReadExternalStorage() {
        String permission = Manifest.permission.READ_EXTERNAL_STORAGE;
        Activity activity = getActivity();
        if (Build.VERSION.SDK_INT == Build.VERSION_CODES.M) {
            if (ContextCompat.checkSelfPermission(
                    activity, permission) != PackageManager.PERMISSION_GRANTED) {
                // Explain to the user why we need to read the contacts (NOT NEEDED!)
                //if (ActivityCompat.shouldShowRequestPermissionRationale(activity,
                //        Manifest.permission.READ_EXTERNAL_STORAGE)) {}
                ActivityCompat.requestPermissions(getActivity(), new String[]{permission},
                        REQUEST_PERMISSION_READ_EXTERNAL_STORAGE);
            } else {
                startActivityForResult(getPickImageIntent(), REQUEST_IMAGE_PICK);
            }
        } else {
            startActivityForResult(getPickImageIntent(), REQUEST_IMAGE_PICK);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions,
                                           @NonNull int[] grantResults) {
        switch (requestCode) {
            case REQUEST_PERMISSION_READ_EXTERNAL_STORAGE:
                if (grantResults.length > 0
                        && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    startActivityForResult(getPickImageIntent(), REQUEST_IMAGE_PICK);
                } else {
                    //Log.d(TAG, "permission READ_EXTERNAL_STORAGE not granted");
                    showSnackBar("Notwendige Berechtigung nicht erteilt",
                            "permission READ_EXTERNAL_STORAGE not granted");
                }

        }
    }

    private File createImageFile() {
        String timeStamp = new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.GERMANY)
                .format(new Date());
        String imageFileName = "JPEG_" + timeStamp + "_";
        Context context = getContext();
        // the media scanner cannot access the files because they are private to our app
        // this is the right directory as base for relative image paths for the future
        File storageDir = context.getExternalFilesDir(Environment.DIRECTORY_PICTURES);
        File image = null;
        try {
            image = File.createTempFile(
                    imageFileName,  /* prefix */
                    ".jpg",         /* suffix */
                    storageDir      /* directory */
            );
            // Save a file: path for use with ACTION_VIEW intents
            //TODO: maybe relative paths are better, this needs to be tested
            //Log.d(TAG, "path="+image.getPath());
            //Log.d(TAG, "absolutePath="+image.getAbsolutePath());
            //Log.d(TAG, image.getPath().substring(image.getPath().indexOf("data/")));
            //Log.d(TAG, Environment.getDataDirectory().getAbsolutePath());
            mCurrentPicturePath = "file:" + image.getAbsolutePath();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return image;
    }

    private void showSnackBar(String snackBarMsg, String fallBackMsg) {
        View view = getView();
        if (view != null) {
            Snackbar.make(view, snackBarMsg, Snackbar.LENGTH_LONG).show();
        } else {
            Log.w(TAG, fallBackMsg);
        }
    }

    private boolean deviceHasCamera() {
        return getActivity().getPackageManager().hasSystemFeature(PackageManager.FEATURE_CAMERA);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            Context context = getContext();
            int dimen;
            if (context.getResources().getConfiguration().orientation
                    == Configuration.ORIENTATION_PORTRAIT) {
                dimen = R.dimen.imageview_detail_dimen_portrait;
            } else {
                dimen = R.dimen.imageview_detail_dimen_landscape;
            }
            if (!mIsUpdate) {
                mPerson = new Person();
            }
            if (requestCode == REQUEST_IMAGE_CAPTURE) {
                /**
                 * this would give us a thumbnail
                 *
                Bundle extras = data.getExtras();
                Bitmap imageBitmap = (Bitmap) extras.get("data");
                mImageViewPicture.setImageBitmap(imageBitmap);
                */
                deleteOldImgFile();// so the old file will not be unused, but this needs be checked by asking the user
                mPerson.setPictureUrl(mCurrentPicturePath);
                //TODO: this also should be the loadRounded method...
                //loadRoundedPictureIntoImageView(context, mCurrentPicturePath, dimen);
                Picasso.with(context)
                        .load(mCurrentPicturePath)
                        .placeholder(R.drawable.user_add)
                        .error(R.drawable.user_add)
                        .transform(new RoundedTransformation(0, dimen))
                        .resizeDimen(dimen, dimen)
                        .centerCrop()
                        .into(mImageViewPicture);
                new UpdatePersonInDb(context, this).execute(mPerson);
            } else if (requestCode == REQUEST_IMAGE_PICK) {
                /**
                 * see: https://github.com/iPaulPro/aFileChooser/blob/master/aFileChooser/src/com/ipaulpro/afilechooser/utils/FileUtils.java
                 *
                 * pictures from gallery:   content://media/external/images/media/2527                      -> document id: null
                 * pictures from recent:    com.android.providers.media.documents/document/image%3A2540     -> document id: image:2540
                 * pictures from camera:    content://media/external/images/media/2402                      -> -"-
                 * pictures from whatsapp:  -"-
                 */
                Uri selectedImageUri = data.getData();
                //TODO
                Log.d(TAG, "getPath="+selectedImageUri.getPath());
                Log.d(TAG, "toString=" + selectedImageUri.toString());

                if (DocumentsContract.isDocumentUri(context, selectedImageUri)) {
                    String docId = DocumentsContract.getDocumentId(selectedImageUri);
                    Log.d(TAG, "documentId="+docId);
                    if (docId.isEmpty()) {
                        Log.d(TAG, "documentId was null although its a documentUri");
                        setPictureWithNonDocumentUri(context, selectedImageUri.toString(), dimen);
                    } else {
                        String[] split = docId.split(":");
                        final String type = split[0];
                        if ("image".equalsIgnoreCase(type)) {
                            getContentFromUri(context, split[1],
                                    MediaStore.Images.Media.EXTERNAL_CONTENT_URI, dimen);
                        } else if ("primary".equalsIgnoreCase(type)) {
                            getContentFromUri(context, split[1],
                                    MediaStore.Images.Media.INTERNAL_CONTENT_URI, dimen);
                        }
                    }
                } else {
                    //Log.d(TAG, "picture with path=" + selectedImageUri + " has no document uri");
                    String realPath = getContentFromNonDocumentIdUri(context, selectedImageUri);
                    if (realPath != null) {
                        //Log.d(TAG, "getContentFromNonDocumentIdUri result="+realPath);
                        //realPath = FILE_PREFIX + realPath;
                        mPerson.setPictureUrl(realPath);
                        loadRoundedPictureIntoImageView(context, realPath, dimen);
                        new UpdatePersonInDb(context, this).execute(mPerson);
                    }/* else {
                        Toast.makeText(context, "Bild hat keine documentUri, Pfad leer", Toast.LENGTH_LONG).show();
                    }*/
                }
            }
        }
    }

    private void loadRoundedPictureIntoImageView(Context context, String path, int dimen) {
        //Picasso.with(context).setLoggingEnabled(true);
        File file = null;
        if (path != null) {
            file = new File(path);
        }
        Picasso.with(context)
                .load(file)
                .placeholder(R.drawable.user_add)
                .error(R.drawable.user_add)
                .transform(new RoundedTransformation(0, dimen))
                .resizeDimen(dimen, dimen)
                .centerCrop()
                .into(mImageViewPicture);
        //mImageViewPicture.invalidate();
        //Picasso.with(context).setLoggingEnabled(false);
    }

    private String getContentFromNonDocumentIdUri(Context context, Uri selectedImageUri) {
        Cursor cursor = null;
        try {
            String[] proj = { MediaStore.Images.Media.DATA };
            cursor = context.getContentResolver().query(selectedImageUri, proj, null, null, null);
            int column_index;
            if (cursor != null) {
                column_index = cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA);
                cursor.moveToFirst();
                return cursor.getString(column_index);
            } else {
                return null;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /**
     * Method to get the absolute path from a given content uri.
     * @param context needed context
     * @param selectionArgs the documentId of the image
     * @param contentUri either MediaStore.Images.Media.EXTERNAL_CONTENT_URI or MediaStore.Images.Media.INTERNAL_CONTENT_URI
     * @param dimen dimen for the Picasso {@link RoundedTransformation}
     */
    private void getContentFromUri(Context context, String selectionArgs, Uri contentUri,
                                   int dimen) {
        String selection = "_id=?";
        String realPath = getDataColumn(context, contentUri, selection,
                new String[] { selectionArgs });
        if (realPath != null) {
            //Log.d(TAG, "getDataColumn result="+realPath);
            String parsedUri = realPath.replace("%2F", "/");
            //parsedUri = FILE_PREFIX + parsedUri;
            mPerson.setPictureUrl(parsedUri);
            loadRoundedPictureIntoImageView(context, realPath, dimen);
            new UpdatePersonInDb(context, this).execute(mPerson);
        } else {
            String s = Environment.getExternalStorageDirectory()+"/"+selectionArgs;
            Log.d(TAG, "getExternalStorageDirectory="+s);
            setPictureWithNonDocumentUri(context, s, dimen);
        }
    }

    private void setPictureWithNonDocumentUri(Context context, String path, int dimen) {
        //Log.d(TAG, path);
        //path = FILE_PREFIX + path;
        mPerson.setPictureUrl(path);
        loadRoundedPictureIntoImageView(context, path, dimen);
        new UpdatePersonInDb(context, this).execute(mPerson);
    }

    /**
     * Helper method for the ContentResolver query. It queries for the absolute path of a file which
     * originally has a "content://"-path.
     * @param context needed context for the getContentResolver()
     * @param uri uri of the file
     * @param selection query selection
     * @param selectionArgs query selection args
     * @return the absolute system path of the file.
     */
    private String getDataColumn(Context context, Uri uri, String selection,
                                String[] selectionArgs) {
        Cursor cursor = null;
        final String column = MediaStore.Images.Media.DATA;
        final String[] projection = { column };
        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs,
                    null);
            if (cursor != null && cursor.moveToFirst()) {
                final int column_index = cursor.getColumnIndexOrThrow(column);
                return cursor.getString(column_index);
            }
        } finally {
            if (cursor != null)
                cursor.close();
        }
        return null;
    }

    private void deleteOldImgFile() {
        if (mPerson != null) {
            if (mPerson.getPictureUrl() != null) {
                int index = mPerson.getPictureUrl().indexOf("/storage");
                if (index > -1) {
                    File file = new File(mPerson.getPictureUrl().substring(index));
                    boolean isDeleted = file.delete();
                    if (!isDeleted) {
                        View view = getView();
                        if (view != null) {
                            Snackbar.make(view, "Konnte Bild nicht löschen", Snackbar.LENGTH_LONG)
                                    .show();
                        } else {
                            Log.w(TAG, "file could not be deleted");
                        }
                    }
                }/* else {
                    Log.i(TAG, "no self made picture, so nothing to delete");
                }*/
            }
        }
    }

    @Override
    public void onUpdate(Person person) {
        mPerson = person;
    }
}
